<?php
namespace Services;

use Utils\Database;

class AuthService {
    public static function registerUser($name, $email, $password, $role, $clientId = null) {
        $db = Database::getInstance();

        // Check if user already exists
        $existingUser = $db->fetch("SELECT id FROM users WHERE email = ?", [$email]);
        if ($existingUser) {
            throw new \Exception("User with this email already exists");
        }

        // Hash password
        $hashedPassword = password_hash($password, PASSWORD_BCRYPT);

        // Get role ID
        $roleData = $db->fetch("SELECT id FROM roles WHERE name = ?", [$role]);
        if (!$roleData) {
            throw new \Exception("Invalid role specified");
        }
        $roleId = $roleData['id'];

        // Insert user
        $userId = $db->execute(
            "INSERT INTO users (name, email, password, role_id, client_id) VALUES (?, ?, ?, ?, ?)",
            [$name, $email, $hashedPassword, $roleId, $clientId]
        );

        return $userId;
    }

    public static function loginUser($email, $password) {
        $db = Database::getInstance();

        // Get user by email
        $user = $db->fetch("SELECT * FROM users WHERE email = ?", [$email]);
        if (!$user) {
            throw new \Exception("Invalid email or password");
        }

        // Verify password
        if (!password_verify($password, $user['password'])) {
            throw new \Exception("Invalid email or password");
        }

        // Get user role
        $role = $db->fetch("SELECT name FROM roles WHERE id = ?", [$user['role_id']]);

        return [
            'id' => $user['id'],
            'name' => $user['name'],
            'email' => $user['email'],
            'role' => $role['name'],
            'client_id' => $user['client_id']
        ];
    }

    public static function getUserById($userId) {
        $db = Database::getInstance();

        $user = $db->fetch("SELECT * FROM users WHERE id = ?", [$userId]);
        if (!$user) {
            return null;
        }

        $role = $db->fetch("SELECT name FROM roles WHERE id = ?", [$user['role_id']]);

        return [
            'id' => $user['id'],
            'name' => $user['name'],
            'email' => $user['email'],
            'role' => $role['name'],
            'client_id' => $user['client_id']
        ];
    }

    public static function checkPermission($userId, $requiredRole) {
        $user = self::getUserById($userId);
        if (!$user) {
            return false;
        }

        return $user['role'] === $requiredRole;
    }
}
